// Background service worker that doesn't require a popup
chrome.action.onClicked.addListener(async (tab) => {
  if (!tab.url.includes('notebooklm.google.com')) {
    chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icon48.png',
      title: 'NotebookLM Chat Exporter',
      message: 'Please open a NotebookLM page first'
    });
    return;
  }

  try {
    // Show a simple dialog to choose format
    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: showExportDialog,
    });
  } catch (error) {
    console.error('Export error:', error);
    chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icon48.png',
      title: 'Export Error',
      message: error.message
    });
  }
});

// This runs in the page context
function showExportDialog() {
  // Create a simple modal dialog
  const dialog = document.createElement('div');
  dialog.style.cssText = `
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    padding: 30px;
    border-radius: 8px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.3);
    z-index: 999999;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    min-width: 300px;
  `;
  
  dialog.innerHTML = `
    <h2 style="margin: 0 0 20px 0; color: #202124;">LogAir.ai: Export Chat</h2>
    <label style="display: block; margin-bottom: 10px; color: #5f6368;">Choose format:</label>
    <select id="exportFormat" style="width: 100%; padding: 8px; margin-bottom: 20px; border: 1px solid #dadce0; border-radius: 4px;">
      <option value="html">HTML (.html)</option>
      <option value="txt">Text (.txt)</option>
      <option value="md">Markdown (.md)</option>
      <option value="json">JSON (.json)</option>
    </select>
    <div style="display: flex; gap: 10px;">
      <button id="exportBtn" style="flex: 1; padding: 10px; background: #1a73e8; color: white; border: none; border-radius: 4px; cursor: pointer; font-weight: 500;">Export</button>
      <button id="cancelBtn" style="flex: 1; padding: 10px; background: #dadce0; color: #202124; border: none; border-radius: 4px; cursor: pointer; font-weight: 500;">Cancel</button>
    </div>
    <div id="status" style="margin-top: 15px; text-align: center; color: #5f6368; font-size: 14px;"></div>
  `;
  
  // Add backdrop
  const backdrop = document.createElement('div');
  backdrop.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.5);
    z-index: 999998;
  `;
  
  document.body.appendChild(backdrop);
  document.body.appendChild(dialog);
  
  const exportBtn = dialog.querySelector('#exportBtn');
  const cancelBtn = dialog.querySelector('#cancelBtn');
  const formatSelect = dialog.querySelector('#exportFormat');
  const statusDiv = dialog.querySelector('#status');
  
  cancelBtn.onclick = () => {
    backdrop.remove();
    dialog.remove();
  };
  
  exportBtn.onclick = async () => {
    exportBtn.disabled = true;
    statusDiv.textContent = 'Extracting chat...';
    
    try {
      const format = formatSelect.value;
      const chatData = extractChat();
      
      if (!chatData || chatData.messages.length === 0) {
        throw new Error('No chat messages found');
      }
      
      const { content, filename, mimeType } = formatData(chatData, format);
      downloadFile(content, filename, mimeType);
      
      statusDiv.style.color = '#34a853';
      statusDiv.textContent = `✓ Exported ${chatData.messages.length} messages`;
      
      setTimeout(() => {
        backdrop.remove();
        dialog.remove();
      }, 1500);
      
    } catch (error) {
      statusDiv.style.color = '#d93025';
      statusDiv.textContent = `Error: ${error.message}`;
      exportBtn.disabled = false;
    }
  };
  
  function extractChat() {
    const messages = [];
    const messagePairs = document.querySelectorAll('.chat-message-pair');
    const maxMessages = Math.min(messagePairs.length, 500);
    
    // Get the notebook title
    let notebookTitle = 'LogAir.ai: NotebookLM Chat';
    const titleElement = document.querySelector('.title-label-inner, editable-project-title .title-label-inner, .notebook-title');
    if (titleElement) {
      notebookTitle = titleElement.textContent.trim() || notebookTitle;
    }
    
    for (let i = 0; i < maxMessages; i++) {
      const pair = messagePairs[i];
      
      const userCard = pair.querySelector('.from-user-message-card-content');
      if (userCard) {
        const userText = userCard.querySelector('.message-text-content');
        if (userText) {
          const content = userText.innerText || userText.textContent || '';
          if (content.trim()) {
            messages.push({
              role: 'user',
              content: content.trim(),
              timestamp: new Date().toISOString()
            });
          }
        }
      }
      
      const assistantCard = pair.querySelector('.to-user-message-card-content');
      if (assistantCard) {
        const assistantText = assistantCard.querySelector('.message-text-content');
        if (assistantText) {
          const clone = assistantText.cloneNode(true);
          const citationButtons = clone.querySelectorAll('.citation-marker');
          citationButtons.forEach(btn => btn.remove());
          
          const contentText = clone.innerText || clone.textContent || '';
          const contentHTML = clone.innerHTML || '';
          
          if (contentText.trim()) {
            messages.push({
              role: 'assistant',
              content: contentText.trim(),
              contentHTML: contentHTML,
              timestamp: new Date().toISOString()
            });
          }
        }
      }
    }
    
    return {
      exportedAt: new Date().toISOString(),
      source: 'NotebookLM',
      url: window.location.href,
      title: notebookTitle,
      messages: messages
    };
  }
  
  function formatData(data, format) {
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
    const safeTitle = (data.title || 'chat').replace(/[^a-zA-Z0-9-_]/g, '_').substring(0, 50);
    
    if (format === 'json') {
      return {
        content: JSON.stringify(data, null, 2),
        filename: `${safeTitle}-${timestamp}.json`,
        mimeType: 'application/json'
      };
    } else if (format === 'md') {
      return {
        content: formatAsMarkdown(data),
        filename: `${safeTitle}-${timestamp}.md`,
        mimeType: 'text/markdown'
      };
    } else if (format === 'html') {
      return {
        content: formatAsHTML(data),
        filename: `${safeTitle}-${timestamp}.html`,
        mimeType: 'text/html'
      };
    } else {
      return {
        content: formatAsText(data),
        filename: `${safeTitle}-${timestamp}.txt`,
        mimeType: 'text/plain'
      };
    }
  }
  
  function formatAsText(data) {
    let txt = `${data.title || 'LogAir.ai: NotebookLM Chat Export'}\n${'='.repeat(50)}\n\n`;
    txt += `Exported: ${new Date(data.exportedAt).toLocaleString()}\n`;
    txt += `Source: ${data.url}\n`;
    txt += `Messages: ${data.messages.length}\n\n${'='.repeat(50)}\n\n`;
    
    data.messages.forEach(msg => {
      txt += `[${msg.role.toUpperCase()}]\n${msg.content}\n\n${'-'.repeat(50)}\n\n`;
    });
    
    return txt;
  }
  
  function formatAsMarkdown(data) {
    let md = `# ${data.title || 'LogAir.ai: NotebookLM Chat Export'}\n\n`;
    md += `**Exported:** ${new Date(data.exportedAt).toLocaleString()}\n`;
    md += `**Source:** ${data.url}\n`;
    md += `**Messages:** ${data.messages.length}\n\n---\n\n`;
    
    data.messages.forEach(msg => {
      const roleEmoji = msg.role === 'user' ? '👤' : '🤖';
      md += `### ${roleEmoji} ${msg.role.toUpperCase()}\n\n${msg.content}\n\n---\n\n`;
    });
    
    return md;
  }
  
  function formatAsHTML(data) {
    const pageTitle = 'LogAir.ai: ' + data.title || 'LogAir.ai: NotebookLM Chat Export';
    let html = `<!DOCTYPE html><html lang="en"><head><meta charset="UTF-8">
    <title>${pageTitle}</title><style>
    body{font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif;max-width:900px;margin:0 auto;padding:20px;background:#f5f5f5;line-height:1.6}
    .header{background:white;padding:30px;border-radius:8px;margin-bottom:20px;box-shadow:0 2px 4px rgba(0,0,0,0.1)}
    .message{background:white;padding:20px;margin-bottom:15px;border-radius:8px;box-shadow:0 2px 4px rgba(0,0,0,0.1)}
    .message-header{font-weight:600;margin-bottom:10px;display:flex;align-items:center;gap:8px}
    .user-badge{background:#1a73e8;color:white;padding:4px 12px;border-radius:4px;font-size:12px;text-transform:uppercase}
    .assistant-badge{background:#34a853;color:white;padding:4px 12px;border-radius:4px;font-size:12px;text-transform:uppercase}
    table{border-collapse:collapse;width:100%;margin:15px 0;box-shadow:0 1px 3px rgba(0,0,0,0.1)}
    table th,table td{border:1px solid #dadce0;padding:12px;text-align:left}
    table th{background:#f1f3f4;font-weight:600}
    table tr:nth-child(even){background:#f8f9fa}
    </style></head><body><div class="header"><h1>${pageTitle}</h1>
    <div>Exported: ${new Date(data.exportedAt).toLocaleString()}</div>
    <div>Messages: ${data.messages.length}</div></div><div class="messages">`;
    
    data.messages.forEach(msg => {
      const badge = msg.role === 'user' ? 'user-badge' : 'assistant-badge';
      const icon = msg.role === 'user' ? '👤' : '🤖';
      const content = msg.contentHTML && msg.role === 'assistant' ? 
        msg.contentHTML : msg.content.replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/\n/g, '<br>');
      html += `<div class="message"><div class="message-header"><span>${icon}</span>
      <span class="${badge}">${msg.role}</span></div><div>${content}</div></div>`;
    });
    
    html += `</div>`;
    html += disclaimer();
    html += `</body></html>`;

    return html;
  }
  
  function disclaimer(html) {
    return `<div><p><span style="color: gray;">
       Artificial Intelligence (AI) systems are not recognized as authoritative sources for airworthiness determinations.
       AI-generated information may contain errors, omissions, or inaccuracies. While AI tools may assist in data interpretation 
       or documentation review, no airworthiness decision, inspection finding, or maintenance release shall be based solely on AI-generated output.

       All determinations affecting aircraft airworthiness must be verified through approved data, official manufacturer publications, 
       and applicable FAA regulations or guidance material.
       </span></p></div>`;
  }
  

  function downloadFile(content, filename, mimeType) {
    const blob = new Blob([content], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }
}
